#!/bin/bash
# File: install_scanner_sdk.sh
#
# Description: DERMALOG software installation script
#
# Author: DERMALOG
#
# Target: linux (RH,SuSE, Debian)
# 
###########################################################################################

show_help(){
	echo -e "\n${SDK_NAME} ${VERSIONINFO}\n"
	echo "Usage: $0 [option]" 
	echo "options:"
	echo "  -m32         Install 32bit"
	echo "  -m64         Install 64bit"
	echo "  -h | --help  Show this help"
	echo ""
}

CURDIR="`dirname $0`"
SDK_CONFIG_NAME="sdk.config"
SDK_CONFIG_PATH="${CURDIR}/$SDK_CONFIG_NAME"

. ${SDK_CONFIG_PATH}

#### Set install architecture ####################################################################
MACHINE_TYPE=`uname -m`
if [[ "$MACHINE_TYPE" == x86_64 ]]; then
	SDK_ARCHITECTURE="64"
else
	SDK_ARCHITECTURE="32"
fi

# user achitecture override
if [ $# -ne 0 ]; then
	case "$1" in
		-m32) SDK_ARCHITECTURE="32" ;;
		-m64) SDK_ARCHITECTURE="64" ;;
		-h|--help) show_help; exit 0 ;;		
		*)
			show_help; 
			echo Invalid option: $1
			exit 1 
			;;
	esac	
fi

#### Setup env ####################################################################
REQLIBS="libusb libusb-1.0"
JAVASDK_FILE="java/DermalogJavaSDK.$JAVASDK_VERSION.tar.gz"
PREFIX="/usr/local"
DERMALOG_HOME="$PREFIX/dermalog"
DERMALOG_INSTALL="$DERMALOG_HOME/x86"
SCANNER_GROUP="dermalog"
BASEDIR="`dirname $0`"
UNINSTALL_SCRIPT=uninstall-scanner-sdk.sh
LIBDIR_32=/usr/lib
LIBDIR_64=/usr/lib64
LIBDIR_64_MULTI=/usr/lib/x86_64-linux-gnu
LIBDIR=${LIBDIR_32}
KERNEL="2.6"
UDEV="/etc/udev"
DERMALOG_LD_CONF="/etc/ld.so.conf.d/dermalog.conf"
DIST=
ERROR=0
DERMALOG_CACHE_DIR="/var/cache/dermalog-fingerprint-scanner-sdk-$VERSIONINFO-x$SDK_ARCHITECTURE"
CONFIG_SCRIPT=config-scanner-sdk.sh
CONFIG_NAME="config.scannersdk.dat"
CONFIG_DAT="$DERMALOG_CACHE_DIR/$CONFIG_NAME"
DERMALOG_ENV="/etc/profile.d/dermalog_scanner_env${SDK_ARCHITECTURE}.sh"

#### Create Temp files ####################################################################
GR_TEMP=/tmp/dermalog_fps_gr.$$
LK_TEMP=/tmp/dermalog_fps_lk.$$
DR_TEMP=/tmp/dermalog_fps_dr.$$
FL_TEMP=/tmp/dermalog_fps_fl.$$
LB_TEMP=/tmp/dermalog_fps_lb.$$

###########################################################################################
#
# Check ROOT permissions
# 
###########################################################################################
check_permissions(){
	if [[ $(id -u) != 0 ]];  then
		echo -e "Root permissions required!\nAbort installation ..."
		exit 1
	fi
}

###########################################################################################
#
# Check licence agreement
# 
###########################################################################################
licenceAgreement() {
	echo -n "Did you read and agree Dermalog Software Licence Agreement? [y/n]"
	read LIC_AGREED
	if [[ $LIC_AGREED != "y" ]]
	then
		echo "Abort installation ... "
		exit 1
	fi
}

###########################################################################################
#
# Check Linux Distribution
# 
###########################################################################################
checkLinuxDist() {
	if [ -f /etc/debian_version ]
	then
		DIST="debian"
	fi
	if [ -f /etc/SuSE-release ] 
	then	
		DIST="suse"
	fi

	if [ -f /etc/redhat-release ]
	then	
		DIST="redhat"
	fi
}

#############################################################################################
#
# Check library path
#
#############################################################################################

setupDefaultLibraryPath(){
	if [ $SDK_ARCHITECTURE -eq 64 ]
	then
		if [ -e $LIBDIR_64_MULTI ]
		then
			LIBDIR=$LIBDIR_64_MULTI
		else
			LIBDIR=$LIBDIR_64
		fi
	else
		LIBDIR=${LIBDIR_32}
	fi
}

#############################################################################################
#
# Check installed Kernel Version
#
#############################################################################################

checkInstalledKernel() {
        KERNEL_RELEASE="`uname -r|awk -F. '{print $1,$2}'|tr -d ' '`"
        KERNEL_NUM="`echo $KERNEL |tr -d '.'`"
        if [ $KERNEL_RELEASE -lt $KERNEL_NUM ];
        then
                echo "ERROR: Kernel version >= $KERNEL is required!"
                exit 1
        fi
}

#############################################################################################
#
# Check Required Libs
#
#############################################################################################
checkRequiredLibs(){
	
	for lib in $REQLIBS
	do
		echo -n "Check $lib installation...."
		if [ $DIST = "debian" ]
		then
			dpkg -l |grep $lib >/dev/null
			ERROR=$?
		
		fi

		if [ $DIST = "suse" ] || [ $DIST = "redhat" ] 
		then
			rpm -q  $lib >/dev/null
			ERROR=$?
		
		fi

		if ! [ $ERROR -eq 0 ]
		then
			echo "Failed."
			echo "Please Install $lib first, then try to install again!"
			exit $ERROR
		else
			echo "OK."
		fi
	done
	
}

##########################################################################################
#
# Setup DERMALOG environment.
#
##########################################################################################
setupEnvironment() {
	echo -n "Please insert installation directory: [$DERMALOG_HOME]"
	read NEW_DERMALOG_HOME
	DERMALOG_HOME="${NEW_DERMALOG_HOME:-$DERMALOG_HOME}"
	
	if [ $SDK_ARCHITECTURE -eq 64 ]
	then
		DERMALOG_INSTALL="$DERMALOG_HOME/x64"
	else
		DERMALOG_INSTALL="$DERMALOG_HOME/x86"
	fi

	DERMALOG_LIB="$DERMALOG_INSTALL/lib"
	DERMALOG_BIN="$DERMALOG_INSTALL/bin"
	DERMALOG_JAVA="$DERMALOG_INSTALL/"
	DERMALOG_RULES="$DERMALOG_INSTALL/rules"
	DERMALOG_DOC="$DERMALOG_INSTALL/doc"
	DERMALOG_TUTORIAL="$DERMALOG_INSTALL/tutorials/VC3"
	DERMALOG_INCLUDE="$DERMALOG_INSTALL/include"
	#
	mkdir -p "$DERMALOG_INSTALL"
	mkdir -p "$DERMALOG_LIB"
	echo "$DERMALOG_LIB" >>"$DR_TEMP"
	mkdir -p "$DERMALOG_BIN"
	echo "$DERMALOG_BIN" >>"$DR_TEMP"
	mkdir -p "$DERMALOG_JAVA"
	echo "$DERMALOG_JAVA" >>"$DR_TEMP"
#	mkdir -p $DERMALOG_RULES
#	echo $DERMALOG_RULES >>$DR_TEMP
	mkdir -p "$DERMALOG_DOC"
	echo "$DERMALOG_DOC" >>"$DR_TEMP"
	mkdir -p "$DERMALOG_TUTORIAL"
	echo "$DERMALOG_TUTORIAL" >>"$DR_TEMP"
	mkdir -p "$DERMALOG_CACHE_DIR"
	echo "$DERMALOG_CACHE_DIR" >> "$DR_TEMP"
	mkdir -p "$DERMALOG_INCLUDE"
	echo "$DERMALOG_INCLUDE" >>"$DR_TEMP"

	echo -n "Please insert group to use DERMALOG's Fingerprint scanner [$SCANNER_GROUP]"
	read NEW_SCANNER_GROUP
	SCANNER_GROUP=${NEW_SCANNER_GROUP:-$SCANNER_GROUP}
	echo $SCANNER_GROUP >> $GR_TEMP
}

#########################################################################################
#
# Install Rules
#
#########################################################################################
Install_udevRules() {
	RULES=/etc/udev/rules.d/80-DERMALOG-scanner.rules
	if [ -f /etc/debian_version ]; then
		 echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ATTRS{idVendor}==\"1491\", ATTRS{idProduct}==\"9860\", MODE=\"0660\", GROUP=\"$SCANNER_GROUP\", SYMLINK+=\"dermalog_zf1_\$env{BUSNUM}_\$env{DEVNUM}\"">$RULES 
		 echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0013\", MODE=\"0660\", GROUP=\"$SCANNER_GROUP\", SYMLINK+=\"dermalog_f1_\$env{BUSNUM}_\$env{DEVNUM}\"">>$RULES 
		 echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0012\", MODE=\"0660\", GROUP=\"$SCANNER_GROUP\", SYMLINK+=\"dermalog_zf2_\$env{BUSNUM}_\$env{DEVNUM}\"">>$RULES 
		 echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0025\", MODE=\"0660\", GROUP=\"$SCANNER_GROUP\", SYMLINK+=\"dermalog_lf1_\$env{BUSNUM}_\$env{DEVNUM}\"">>$RULES 
		 echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0011\", MODE=\"0660\", GROUP=\"$SCANNER_GROUP\", SYMLINK+=\"dermalog_lf10_\$env{BUSNUM}_\$env{DEVNUM}\"">>$RULES 
	else
		echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ENV{DEVTYPE}==\"usb_device\", ATTRS{idVendor}==\"1491\", ATTRS{idProduct}==\"9860\", NAME=\"bus/usb/\$env{BUSNUM}/\$env{DEVNUM}\",MODE=\"0660\", GROUP=\"$SCANNER_GROUP\"" >$RULES
		echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ENV{DEVTYPE}==\"usb_device\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0013\", NAME=\"bus/usb/\$env{BUSNUM}/\$env{DEVNUM}\",MODE=\"0660\", GROUP=\"$SCANNER_GROUP\"" >>$RULES
		echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ENV{DEVTYPE}==\"usb_device\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0012\", NAME=\"bus/usb/\$env{BUSNUM}/\$env{DEVNUM}\",MODE=\"0660\", GROUP=\"$SCANNER_GROUP\"" >>$RULES
		echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ENV{DEVTYPE}==\"usb_device\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0025\", NAME=\"bus/usb/\$env{BUSNUM}/\$env{DEVNUM}\",MODE=\"0660\", GROUP=\"$SCANNER_GROUP\"" >>$RULES
		echo "SUBSYSTEM==\"usb*\", ACTION==\"add\", ENV{DEVTYPE}==\"usb_device\", ATTRS{idVendor}==\"1fba\", ATTRS{idProduct}==\"0011\", NAME=\"bus/usb/\$env{BUSNUM}/\$env{DEVNUM}\",MODE=\"0660\", GROUP=\"$SCANNER_GROUP\"" >>$RULES
	fi
	echo $RULES>>"$FL_TEMP"
	
	which udevcontrol >/dev/null 2>&1 
	if [[ $? = 0 ]]; then
		udevcontrol reload_rules
	else
		which udevadm >/dev/null 2>&1
		if [[ $? = 0 ]]; then
			udevadm control --reload-rules >/dev/null 2>&1
		else
			echo "Please reload udevrules manually. The tools udevcontrol or udevadm are not found in the system"
		fi
	fi
}

##########################################################################################
#
# Install Package
#
##########################################################################################
InstallPkg() {

	groupadd $SCANNER_GROUP >/dev/null
	DIST_DRM_LIB="${CURDIR}/lib${SDK_ARCHITECTURE}"
	DIST_DRM_BIN="${CURDIR}/bin${SDK_ARCHITECTURE}"
	DIST_DRM_RULES="$CURDIR/rules"
	DIST_DRM_DOC="$CURDIR/doc"
	DIST_DRM_INCLUDE="$CURDIR/include"	
	DIST_DRM_JAVA="$CURDIR/$JAVASDK_FILE"
	
	echo "Installing libraries..."
	
	# Coping all libs from the source to the distination directory.
	LIBS="`ls \"$DIST_DRM_LIB/\" |xargs`" 
	for lib in $LIBS
	do
		cp "$DIST_DRM_LIB/$lib" "$DERMALOG_LIB/$lib"
		echo "$DERMALOG_LIB/$lib" >> "$LB_TEMP"
		echo "$DERMALOG_LIB/$lib" >> "$FL_TEMP"

		LINKNAME="`echo $lib|grep -e [a-zA-Z].*\.so\.[0-9]\.[0-9]|awk -F.  '{print $1,$2,$3}' |tr -t ' ' '.'`"
		ALTERNATIV_LINK="`echo $lib|grep -e 'gcc\.Release\.[0-9]*\.so'|awk -F.  '{print $1}'`".so
		if ! [ $ALTERNATIV_LINK = '.so' ]; then
			LINKNAME="$ALTERNATIV_LINK"
		fi

		if ! [ -z $LINKNAME ]; then
			ln -sf "$DERMALOG_LIB/$lib"  "$LIBDIR/$LINKNAME"
			echo "$LIBDIR/$LINKNAME" >> "$LK_TEMP"
		else
			ln -sf "$DERMALOG_LIB/$lib"  $LIBDIR/$lib
			echo $LIBDIR/$lib >> $LK_TEMP
		fi

		LINKNAME="`echo $lib|grep -e [a-zA-Z].*\.so\.[0-9]\.[0-9]|awk -F. '{print $1,$2}' |tr -t ' ' '.'`"
		if ! [ -z $LINKNAME ]; then
			ln -sf $DERMALOG_LIB/$lib  $LIBDIR/$LINKNAME
			echo $LIBDIR/$LINKNAME >> $LK_TEMP
		fi

		chgrp -R $SCANNER_GROUP $DERMALOG_LIB/
		chmod -R  g+rwx $DERMALOG_LIB
	done
	
	# Coping all executable from bin to the destination directory and create the links
	echo "Installing binaries..."
	EXECS="`ls \"$DIST_DRM_BIN/\" |xargs`"
	for exe in $EXECS
	do
		cp "$DIST_DRM_BIN/$exe" "$DERMALOG_BIN/$exe"
		echo "$DERMALOG_BIN/$exe" >> "$FL_TEMP"
		ln -sf "$DERMALOG_BIN/$exe" "$PREFIX/bin/$exe"
		echo "$PREFIX/bin/$exe" >> "$LK_TEMP"
	done
	chgrp -R $SCANNER_GROUP  "$DERMALOG_BIN/"
	chmod -R ug+rx "$DERMALOG_BIN"
	
	#Coping all java files
	echo "Installing Java wrapper..."
	tar xzf "$DIST_DRM_JAVA" -C "$DERMALOG_JAVA"
	
	# Install Udev Rules
	echo "Installing udev rules..."
	Install_udevRules

	# Coping documents
	echo "Installing doc..."
	DOCS="`ls \"$DIST_DRM_DOC/\" |xargs`"
	for doc in $DOCS
	do
		cp "$DIST_DRM_DOC/$doc" $DERMALOG_DOC/$doc
		echo $DERMALOG_DOC/$doc >> $FL_TEMP
	done
	
	#copy header files
	echo "Installing C header files..."
	INCLUDES="`ls \"$DIST_DRM_INCLUDE/\" | xargs`"
	for inc in $INCLUDES
	do
		cp "$DIST_DRM_INCLUDE/$inc" $DERMALOG_INCLUDE/$inc
		echo $DERMALOG_INCLUDE/$inc >> $FL_TEMP
	done
	
	chgrp -R $SCANNER_GROUP "$DERMALOG_DOC/$doc"
	# Copy lisence 
	cp "$CURDIR/"*.pdf "$DERMALOG_INSTALL" &>/dev/null
	chmod u+xr "$DERMALOG_INSTALL/"*.pdf &>/dev/null
	
	# Copy administration files.	
	cp "$CURDIR/$UNINSTALL_SCRIPT" "$DERMALOG_INSTALL/$UNINSTALL_SCRIPT"
	chmod u+x $DERMALOG_INSTALL/$UNINSTALL_SCRIPT
	cp "$CURDIR/$CONFIG_SCRIPT" "$DERMALOG_INSTALL/$CONFIG_SCRIPT"
	chmod u+x $DERMALOG_INSTALL/$CONFIG_SCRIPT
	#copy and patch SDK config
	cp "$CURDIR/$SDK_CONFIG_NAME" "$DERMALOG_INSTALL/$SDK_CONFIG_NAME"	
	chmod u+x $DERMALOG_INSTALL/$SDK_CONFIG_NAME	
	echo -e "\nSDK_ARCHITECTURE=${SDK_ARCHITECTURE}" >> "$DERMALOG_INSTALL/$SDK_CONFIG_NAME"
}
##########################################################################################
#
# Install LIBRARY_PATH
# 
##########################################################################################
InstallLibraryPath(){
	echo "export LD_LIBRARY_PATH=\"$LD_LIBRARY_PATH:${DERMALOG_LIB}\"" >"${DERMALOG_ENV}"
    echo "${DERMALOG_ENV}" >> ${FL_TEMP}
}
##########################################################################################
#
# Install Package
#
##########################################################################################
IntallTutorial()
{
	cp "$CURDIR/tutorials" -r $DERMALOG_INSTALL
	chgrp -R $SCANNER_GROUP $DERMALOG_INSTALL/tutorials
	chmod -R ug+w $DERMALOG_INSTALL/tutorials
}
##########################################################################################
#
# Create Config.dat.
#
##########################################################################################
createConfig(){
	TEMP_CONFIG_DAT=/tmp/$CONFIG_NAME.$$

	touch $TEMP_CONFIG_DAT
	
	## Append version
	echo "VERSIONINFO=\"$VERSIONINFO\"">>$TEMP_CONFIG_DAT
	
	## Append Dermalog Home
	echo "DERMALOG_INSTALL=$DERMALOG_INSTALL">>$TEMP_CONFIG_DAT
	
	## Append group
	echo "SCANNER_GROUP=\$(cat <<EOG" >>$TEMP_CONFIG_DAT
	cat $GR_TEMP >>$TEMP_CONFIG_DAT
	echo "EOG" >>$TEMP_CONFIG_DAT
	echo ")" >>$TEMP_CONFIG_DAT

	## Append files
	echo "SCANNER_FILES=\$(cat <<EOFS" >>$TEMP_CONFIG_DAT
	cat $FL_TEMP >>$TEMP_CONFIG_DAT
	echo "EOFS" >>$TEMP_CONFIG_DAT
	echo ")" >>$TEMP_CONFIG_DAT

	## Append libs
	echo "SCANNER_LIBS=\$(cat <<EOLS" >>$TEMP_CONFIG_DAT
	cat $LB_TEMP >>$TEMP_CONFIG_DAT
	echo "EOLS" >>$TEMP_CONFIG_DAT	
	echo ")" >>$TEMP_CONFIG_DAT

	## Append links
	echo "SCANNER_LINKS=\$(cat <<EOL" >>$TEMP_CONFIG_DAT
	cat $LK_TEMP >>$TEMP_CONFIG_DAT
	echo "EOL" >>$TEMP_CONFIG_DAT
	echo ")" >>$TEMP_CONFIG_DAT
	
	## Append directories
	echo "SCANNER_DIRS=\$(cat <<EOD" >>$TEMP_CONFIG_DAT
	cat $DR_TEMP >>$TEMP_CONFIG_DAT
	echo "EOD" >>$TEMP_CONFIG_DAT
	echo ")" >>$TEMP_CONFIG_DAT
		
	cp "$TEMP_CONFIG_DAT" $CONFIG_DAT
	## make clean
	rm $TEMP_CONFIG_DAT
	rm $GR_TEMP
	rm $FL_TEMP
	rm $LK_TEMP
	rm $DR_TEMP
}

echo ""
echo "$SDK_NAME $VERSIONINFO ${SDK_ARCHITECTURE}bit"
echo ""
check_permissions
licenceAgreement
checkInstalledKernel
checkLinuxDist
checkRequiredLibs
setupDefaultLibraryPath
setupEnvironment
InstallPkg
InstallLibraryPath
IntallTutorial
createConfig

#remove some unneeded files 
find $DERMALOG_INSTALL/ -name 'Thumbs.db' | xargs rm
#remove empty directories
find $DERMALOG_INSTALL/ -type d -empty | xargs -n 1 rmdir

echo "Installation complete"
echo ""
echo "Please reboot your system or execute the following line as root to update the environment variables"
echo "> source ${DERMALOG_ENV}"
echo ""
echo "Make sure to add your user to the group \"$SCANNER_GROUP\" to access the scanner without root privileges."
echo "> usermod -a -G $SCANNER_GROUP <username>"
echo ""
